<?php
/**
 * This file is part of osCommerce ecommerce platform.
 * osCommerce the ecommerce
 *
 * @link https://www.oscommerce.com
 * @copyright Copyright (c) 2000-2022 osCommerce LTD
 *
 * Released under the GNU General Public License
 * For the full copyright and license information, please view the LICENSE.TXT file that was distributed with this source code.
 */

namespace common\classes\Billing;

use common\helpers\Assert;

class BillingManager
{
    private static $billingPoints = null;

    private static function getBillingPoints()
    {
        if (is_null(self::$billingPoints)) {
            self::$billingPoints = [];
            foreach (\common\helpers\Hooks::getList('billing/billing-points') as $filename) {
                include($filename); // use self::addBillingPoint()
            }
        }
        return self::$billingPoints;
    }

    private static function getBillingPointCheck(string $operation)
    {
        $points = self::getBillingPoints();
        Assert::keyExists($points, $operation, "Billing point not found for $operation");
        return $points[$operation];
    }

    private static function addBillingPoint($billingPointClass)
    {
        Assert::classImplements($billingPointClass, \common\classes\Billing\BillingPointInterface::class);
        foreach ($billingPointClass::getSupportedOperations() as $operation) {
            Assert::assert(!array_key_exists($operation, self::getBillingPoints()), "Billing point already exist for $operation");
            self::$billingPoints[$operation] = $billingPointClass;
        }
    }

    /**
     * @param string $operation 'generate-product-info' / 'translate-product-info'
     * @param string $domainKey domain key generated by installation
     * @param string $appKey AppShop key generated in oscommerce.com
     * @param $extraParams
     * @return void
     */
    public static function checkOperationAvailable(string $operation, string $domainKey, string $appKey, $extraParams = null)
    {
        $res = self::checkOperationAvailableSafe($operation, $domainKey, $appKey, $extraParams);
        if (((isset($res['result']) && $res['result'] === false) || (isset($res['success']) && $res['success'] === false)) && (isset($res['message']) || isset($res['error']))) {
            throw new \yii\base\UserException($res['message'] ?? $res['error'] ?? 'Unknown error');
        }
    }

    public static function checkOperationAvailableSafe(string $operation, string $domainKey, string $appKey, $extraParams = null)
    {
        try {
            $billingPoint = self::getBillingPointCheck($operation);
            return $billingPoint::checkOperationAvailable($operation, $domainKey, $appKey, $extraParams);
        } catch (\Throwable $e) {
            \common\helpers\Php::logError($e, __CLASS__ . '::' . __FUNCTION__);
            return self::getInternalErrorResult();
        }
    }


    public static function doOperationFinished(string $operation, string $domainKey, string $appKey, $extraParams = null)
    {
        try {
            $billingPoint = self::getBillingPointCheck($operation);
            $billingPoint::doOperationFinished($operation, $domainKey, $appKey, $extraParams);
        } catch (\Throwable $e) {
            \common\helpers\Php::logError($e, __CLASS__ . '::' . __FUNCTION__);
        }
    }

    public static function getInternalErrorResult($message = 'Internal error')
    {
        return ['result' => false, 'message' => $message];
    }

}